<?php
/**
 * File: sinkrontransaksi.php
 * Menerima data transaksi dari klien, menyimpannya, dan mengurangi stok.
 */

header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Content-Type: application/json; charset=UTF-8");

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') { exit(); }
function send_json_response($statusCode, $data) { http_response_code($statusCode); echo json_encode($data); exit(); }

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

try {
    // --- Otentikasi (WAJIB ADA) ---
    if (!isset($_GET['token']) || empty($_GET['token'])) { send_json_response(401, ['message' => 'Akses ditolak. Token tidak ditemukan.']); }
    $token = $_GET['token'];

    $conn = new mysqli("localhost", "root", "", "dbkasir");
    $conn->set_charset("utf8mb4");

    $sessionSql = "SELECT * FROM sesi_pengguna WHERE token = ? LIMIT 1";
    $stmt = $conn->prepare($sessionSql);
    $stmt->bind_param("s", $token);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows !== 1) { send_json_response(401, ['message' => 'Sesi tidak valid.']); }
    $sessionData = $result->fetch_assoc();
    if (time() > strtotime($sessionData['kadaluarsa'])) { send_json_response(401, ['message' => 'Sesi kedaluwarsa.']); }
    
    // Ambil id_pengguna dari sesi yang valid untuk digunakan sebagai id_kasir
    $id_kasir = $sessionData['id_pengguna'];
    $stmt->close();
    // --- Akhir Otentikasi ---

    // Pastikan metode adalah POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') { send_json_response(405, ['message' => 'Metode harus POST.']); }
    
    // Ambil dan decode data JSON dari body
    $json_data = file_get_contents('php://input');
    $transactions = json_decode($json_data, true);

    if (json_last_error() !== JSON_ERROR_NONE || !is_array($transactions) || empty($transactions)) {
        send_json_response(400, ['message' => 'Data transaksi tidak valid atau kosong.']);
    }

    $errors = [];
    $success_count = 0;

    // Loop setiap transaksi yang dikirim oleh klien
    foreach ($transactions as $trx) {
        // Gunakan MySQL Transaction untuk memastikan integritas data
        $conn->begin_transaction();
        try {
            // 1. Simpan data ke tabel 'transaksi'
            $sql_transaksi = "INSERT INTO transaksi (nomor_invoice, id_kasir, id_pelanggan, id_lokasi, jumlah_total, jumlah_diskon, metode_pembayaran, status_pembayaran, is_sinkron, dibuat_pada, diperbarui_pada) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1, ?, NOW())";
            $stmt_transaksi = $conn->prepare($sql_transaksi);
            
            // Atasi nilai null untuk id_pelanggan
            $id_pelanggan = isset($trx['id_pelanggan']) && !empty($trx['id_pelanggan']) ? $trx['id_pelanggan'] : null;

            $stmt_transaksi->bind_param("siiiddsss", 
                $trx['nomor_invoice'], $id_kasir, $id_pelanggan, $trx['id_lokasi'], 
                $trx['jumlah_total'], $trx['jumlah_diskon'], $trx['metode_pembayaran'], 
                $trx['status_pembayaran'], $trx['dibuat_pada']
            );
            $stmt_transaksi->execute();
            
            // Ambil ID dari transaksi yang baru saja disimpan
            $new_transaction_id = $conn->insert_id;
            $stmt_transaksi->close();

            // 2. Loop dan simpan setiap item ke 'detail_transaksi' & kurangi stok
            if (isset($trx['details']) && is_array($trx['details'])) {
                foreach ($trx['details'] as $item) {
                    // Simpan detail
                    $sql_detail = "INSERT INTO detail_transaksi (id_transaksi, id_produk, id_varian_produk, kuantitas, harga_per_unit, subtotal) VALUES (?, ?, ?, ?, ?, ?)";
                    $stmt_detail = $conn->prepare($sql_detail);
                    $id_varian_produk = isset($item['id_varian_produk']) && !empty($item['id_varian_produk']) ? $item['id_varian_produk'] : null;
                    $stmt_detail->bind_param("iiiidd", 
                        $new_transaction_id, $item['id_produk'], $id_varian_produk, 
                        $item['kuantitas'], $item['harga_per_unit'], $item['subtotal']
                    );
                    $stmt_detail->execute();
                    $stmt_detail->close();

                    // Kurangi stok
                    $sql_stok = "UPDATE stok SET kuantitas = kuantitas - ? WHERE id_produk = ? AND id_lokasi = ?";
                    $stmt_stok = $conn->prepare($sql_stok);
                    $stmt_stok->bind_param("iii", $item['kuantitas'], $item['id_produk'], $trx['id_lokasi']);
                    $stmt_stok->execute();
                    $stmt_stok->close();
                }
            }
            
            // Jika semua query berhasil, commit transaksi
            $conn->commit();
            $success_count++;
        } catch (Exception $e) {
            // Jika ada satu saja query yang gagal, batalkan semua perubahan untuk transaksi ini
            $conn->rollback();
            $errors[] = "Gagal menyimpan invoice " . ($trx['nomor_invoice'] ?? 'N/A') . ": " . $e->getMessage();
        }
    }
    
    // Kirim respons akhir ke klien
    if (empty($errors)) {
        send_json_response(200, ['message' => "$success_count transaksi berhasil disinkronkan."]);
    } else {
        send_json_response(500, ['message' => "Beberapa transaksi gagal disinkronkan.", 'errors' => $errors]);
    }
    
    $conn->close();

} catch (Throwable $e) { 
    error_log("Sync Transaksi API Error: " . $e->getMessage()); 
    send_json_response(500, ['message' => 'Terjadi kesalahan tidak terduga pada server.']); 
}
?>